/**
 * Copyright (c) 2006 Macacos.org. All Rights Reserved
 *
 * This library is free software; you can redistribute it and/or modify it under the terms of the 
 * GNU Lesser General Public License as published by the Free Software Foundation; either version 
 * 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; 
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See 
 * the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this library;
 * if not, write to the Free Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, 
 * MA  02110-1301  USA
 * 
 * You can find the license also here: http://www.gnu.org/copyleft/lesser.html
 * 
 * 
 * Created on Sep 25, 2006 by Thiago Tonelli Bartolomei
 * -------------------------------------------------
 *           \                                                                           
 *            \                                                                          
 *               __                                                                      
 *          w  c(..)o                                                                    
 *           \__(o)                                                                      
 *               /\                                                                      
 *            w_/(_)-~                                                                   
 *                /|                                                                     
 *               | \                                                                     
 *               m  m   
 */
package healthwatcher.aspects.patterns;

import healthwatcher.business.HealthWatcherFacade;
import healthwatcher.model.complaint.Complaint;
import healthwatcher.model.employee.Employee;
import healthwatcher.model.healthguide.HealthUnit;
import healthwatcher.view.*;
import healthwatcher.view.command.*;
import healthwatcher.view.servlets.*;
import javax.servlet.http.*;

import org.aspectj.lang.SoftException;

import lib.patterns.ObserverProtocol;

/**
 * This aspect implements the observer protocol for the Health Watcher System
 * 
 * @author Thiago Tonelli Bartolomei <thiago.bartolomei@gmail.com>
 */
public deployed cclass UpdateStateObserver extends ObserverProtocol {
	
	// Wrappers for the subjects
	public cclass ComplaintSubject extends Subject wraps Complaint {}
	public cclass EmployeeSubject extends Subject wraps Employee {}
	public cclass HealthUnitSubject extends Subject wraps HealthUnit {}
	
	// Wrappers for the observer
	public cclass FacadeObserver extends Observer wraps IFacade {
		
		public void notify(Subject subject) {
			
			try {
				if(subject instanceof EmployeeSubject){
					wrappee.updateEmployee(((EmployeeSubject) subject).wrappee);
				} else if(subject instanceof ComplaintSubject){
					wrappee.updateComplaint(((ComplaintSubject) subject).wrappee);	
				} else if(subject instanceof HealthUnitSubject){
					wrappee.updateHealthUnit(((HealthUnitSubject) subject).wrappee);
				}
			} catch (Exception e) {
				throw new SoftException(e);
			}
		}
	}
	
	/**
	 * Defines where we apply the advice
	 */
	protected pointcut where() : this(CommandServlet+);
	
	/**
	 * When a value is stored in the session, create a subject and store the observer to it 
	 */
	after(Object o) : where() && call(* HttpSession+.putValue(String, Object)) && args(.., o) {

		Subject s = getSubject(o);
		if (s != null) {
			s.addObserver(FacadeObserver(HealthWatcherFacade.getInstance()));
		}
	}
	
	/**
	 * Determines if the object is a subject
	 */
	protected Subject getSubject(Object o) {
		if (o instanceof Employee) {
			return EmployeeSubject((Employee) o);
		} else if (o instanceof Complaint) {
			return ComplaintSubject((Complaint) o);
		} else if (o instanceof HealthUnit) {
			return HealthUnitSubject((HealthUnit) o);
		}
		return null;
	}
	
	/**
	 * Detect changes
	 */
	after(Employee e) : where() && call(* Employee+.set*(..)) && target(e) {
		EmployeeSubject(e).changed();
	}
	
	after(Complaint c) : where() && call(* Complaint+.set*(..)) && target(c) {
		ComplaintSubject(c).changed();
	}
	
	after(HealthUnit h) : where() && call(* HealthUnit+.set*(..)) && target(h) {
		HealthUnitSubject(h).changed();
	}
}
