// -*- java -*-
//
// $Id: SubstitutionTransition.js 10400 2007-08-07 23:38:00Z knoke $
//
// *************************************
//  SubstitutionTransition
// *************************************

/**
 * Creates a new substitution transition object.
 *
 * @param javaObject The java object to encapsulate.
 */
function SubstitutionTransition(javaObject) {
    if (javaObject == null) {
	session.throwException("SubstitutionTransition constructor: provided javaObject == null!");
    }
    this.javaObject = javaObject;
}


/**
 * Adds a new substitution transition to the given net.
 *
 * @param net The net to create the new substitution transition in.
 * @param name The name of this new substitution transition.
 * @return The new SubstitutionTransition object.
 */
SubstitutionTransition.add = function(net, name) {
  if (net == null)
    session.throwException("SubstitutionTransition.add: provided net == null!");
  if (name == null || name.equals("")) {
    session.throwException("SubstitutionTransition.add: name is invalid (empty or null)!");
  }

  return new SubstitutionTransition(session.addSubstitutionTransition(net.javaObject, name));
}


/**
 * Tries to find the substitution transition with the given name inside the provided net.
 *
 * @param net The net to search in.
 * @param name The name of the transition to search.
 * @return The substitution transition, if found.
 */
SubstitutionTransition.withName = function(net, name) {
  if (net == null)
    session.throwException("No net for SubstitutionTransition.withName()!");
  if (name == null) {
    session.throwException("SubstitutionTransition.withName: provided name == null!");
  }

  return new SubstitutionTransition(session.getSubstitutionTransitionWithName(net.javaObject, name));
}


/**
 * Returns the name of this SubstitutionTransition.
 *
 * @return The name of this transition.
 */
SubstitutionTransition.prototype.getName = function() {
    this.javaObject.getName();
}


/**
 * Returns the ID of this SubstitutionTransition.
 *
 * @return The ID of this transition.
 */
SubstitutionTransition.prototype.getID = function() {
    this.javaObject.getId();
}


/** 
 * Sets the local guard expression for this transition.
 *
 * @param guardStr The new local guard expression.
 */
SubstitutionTransition.prototype.setLocalGuard = function(guardStr) {
    this.javaObject.setLocalGuard(guardStr);
}


/**
 * Returns the actual local guard expression.
 *
 * @return The actual local guard expression in String form.
 */
SubstitutionTransition.prototype.getLocalGuard = function() {
    return this.javaObject.getLocalGuard().toString();
}


/**
 * Sets the global guard expression for this transition.
 *
 * @param guardStr The new global guard expression.
 */
SubstitutionTransition.prototype.setGlobalGuard = function(guardStr) {
    this.javaObject.setGlobalGuard(guardStr);
}


/**
 * Returns the actual global guard expression.
 *
 * @return The actual global guard expression in String form.
 */
SubstitutionTransition.prototype.getGlobalGuard = function() {
    return this.javaObject.getGlobalGuard().toString();
}


/**
 * Returns the number of replications for this SubstitutionTransition.
 *
 * @return The number of Replications
 */
SubstitutionTransition.prototype.getNumberOfReplications = function() {
    return this.javaObject.getNumberOfReplications();
}


/**
 * Returns the n-th Replication object.
 *
 * @param n The number of the Replication to return.
 * @return The according Replication object.
 */
SubstitutionTransition.prototype.getReplication = function(n) {
    return new Replication(this.javaObject.getReplication(n));
}
