// -*- java -*-
//
// $Id: Place.js 10400 2007-08-07 23:38:00Z knoke $
//
// *************************************
//  Place
// *************************************

/**
 * Creates a new JS Place representation.
 * 
 * @param javaObject The java object to encapsulate.
 */
function Place(javaObject) {
    if (javaObject == null) {
	session.throwException("Place constructor: provided javaObject == null!");
    }
    this.javaObject = javaObject;
}


/**
 * Adds a new place to the specified net with the given values.
 *
 * @param net The net to add this place to.
 * @param name The name of the place to add.
 * @param tokentype The allowed type of this place.
 * @param capacity The maximum capacity of this place.
 * @param queue The queue type.
 * @param watch The watch string
 * @param initialMarking The initial marking of the place.
 * @return The newly created Place.
 */
Place.add = function(net, name, tokentype, capacity, queue, watch, initialMarking) {
    if (net == null) {
	session.throwException("The provided net was null..");
    }

    if (name == null || name.equals("")) {
	session.throwException();
    }
    if (tokentype == null) {
	session.throwException();
    }
    if (capacity == null) {
	session.throwException();
    }
    if (queue == null) {
	session.throwException();
    }
    if (watch == null) {
	session.throwException();
    }
    if (initialMarking == null) {
	session.throwException();
    }

    return new Place(session.addPlace(net.javaObject, name, tokentype, capacity, queue, watch, initialMarking));   
}


/**
 * Tries to find the place with the name in the given net.
 *
 * @param net The net to search the place in.
 * @param name The name of the place to search.
 * @return The place, if found.
 */
Place.withName = function(net, name) {
    if (name == null) {
	session.throwException("Place.withName: provided name == null!");
    }
    
    return new Place(session.getPlaceByName(net.javaObject, name));
}


/**
 * Returns the name of this place.
 *
 * @return The name of the place.
 */
Place.prototype.getName = function() {
    return this.javaObject.getName();
}


/**
 * Returns this place's id.
 *
 * @return The id of this place.
 */
Place.prototype.getID = function() {
    return this.javaObject.getId();
}


/**
 * Returns this place's capacity.
 *
 * @return The capacity of this place.
 */
Place.prototype.getCapacity = function() {
    return this.javaObject.getCapacityStr();
}


/**
 * Returns the allowed tokentype of this place.
 *
 * @return The tokentype of this place.
 */
Place.prototype.getTokentype = function() {
    return this.javaObject.getTokentypeStr();
}


/**
 * Returns the queue type.
 *
 * @return The queue type of this place.
 */
Place.prototype.getQueue = function() {
    return this.javaObject.getQueueStr();
}


/**
 * Returns the watch expression of this place.
 *
 * @return The watch of this place.
 */
Place.prototype.getWatch = function() {
    return this.javaObject.getWatchStr();
}


/**
 * Returns the initial marking of this place.
 *
 * @return The place's initial marking.
 */
Place.prototype.getInitialMarking = function() {
    return this.javaObject.getInitialMarkingStr();
}


/**
 * Sets the initial marking of this place.
 *
 * @param initialMarking The new initial marking.
 */
Place.prototype.setInitialMarking = function(initialMarking) {
    if (initialMarking == null) {
	session.throwException("Place.setInitialMarking: Provided initialMarking == null!");
    } else {
	this.javaObject.setInitialMarking(initialMarking);
    }
}


/**
 * Sets the name of this place object.
 *
 * @param name The new name for this place.
 */
Place.prototype.setName = function(name) {
    if (name == null) {
	session.throwException("Place.setName: provided name == null!");
    } else {
	this.javaObject.setName(name);
    }
}


/**
 * Sets the Tokentype of this Place.
 *
 * @param tokentype The name of the tokentype to set.
 */
Place.prototype.setTokentype = function(tokentype) {
    if (tokentype == null) {
	session.throwException("Place.setTokentype: provieded tokentype == null!");
    } else {
	this.javaObject.setTokentype(tokentype);
    }
}


/**
 * Sets the capacity of this place.
 *
 * @param capacity The new capacity value.
 */
Place.prototype.setCapacity = function(capacity) {
    if (capacity == null) {
	session.throwException("Place.setCapacity: provieded capacity == null!");
    } else {
	this.javaObject.setCapacity(capacity);
    }
}


/**
 * Sets the queue of this place.
 *
 * @param queue The new queue value.
 */
Place.prototype.setQueue = function(queue) {
    if (queue == null) {
	session.throwException("Place.setQueue: provieded queue == null!");
    } else {
	this.javaObject.setQueue(queue);
    }
}


/**
 * Sets the watch of this place.
 *
 * @param watch The new watch value.
 */
Place.prototype.setWatch = function(watch) {
    if (watch == null) {
	session.throwException("Place.setWatch: provieded watch == null!");
    } else {
	this.javaObject.setWatch(watch);
    }
}


/**
 * Returns true, if this place equals the provided place.
 *
 * @param otherPlace The other place to use in this equals() operation
 * @return true, if the places are equal, false otherwise.
 */
Place.prototype.equals = function(otherPlace) {
    if (! otherPlace instanceof Place) {
	session.throwException("Place.equals: Given place is not of type place!");
    } 
    
    return this.javaObject.equals(otherPlace.javaObject);
}
    

