// -*- java -*-
//
// $Id: Net.js 10400 2007-08-07 23:38:00Z knoke $
//
// *************************************
//  Net
// *************************************

/** 
 * Creates a new Net object.
 *
 * @param javaObject The java object to encapsulate.
 */
function Net(javaObject) {
    if (javaObject == null) {
	session.throwException("Net constructor: provided javaObject == null!");
    }
    this.javaObject = javaObject;
}


/**
 * Creates a new, empty Net.
 *
 * @return A new Net object.
 */
Net.create = function() {
    return new Net(session.createNet());
}


/**
 * Loads an existing net.
 *
 * @param filename The filename of the net to load.
 * @return A net object initialized with the read in net data.
 */
Net.load = function(filename) {
    return new Net(session.createNet(filename));
}


/**
 * Imports the library specified by the given filename.
 *
 * @param filename The name of the library file to iport.
 */
Net.prototype.importLib = function(filename) {
  if (filename == null)
    session.throwException("No filename for library import.");
  
  // FIXME: What about the LibImport object?
  session.addLibImport(this.javaObject, filename);
}


/**
 * Checks the net for errors.
 * This version emits all errors and throws an exception afterwards.
 */
Net.prototype.check = function() {
    var errs = session.checkNet(this.javaObject);
    if (errs != null && errs.length > 0) {
	for (idx=0; idx < errs.length; idx++) {
	    println("["+idx+"] "+errs[idx]);
	}
	session.throwException("Net check has failed. Please check the error listing.");
    }
}


/**
 * Checks this net for errors.
 * This version returns an array of errors.
 *
 * @return An array of errors.
 */
Net.prototype.check2 = function() {
    return session.checkNet2(this.javaObject);
}


/**
 * Exports this net to the given file.
 *
 * @param filename The name of the file tpo export to.
 * @return The result of this operation (true/false)
 */
Net.prototype.toFile = function(filename) {
    this.check();
    return session.exportNet(this.javaObject, filename);
}


/**
 * Returns an array of all places inside this net.
 *
 * @return An array of all this net's places.
 */
Net.prototype.getPlaces = function() {
    var iterator = this.javaObject.iteratePlaces();
    var placeArray = new Array();
    
    for (var idx=0; iterator.hasNext(); idx++) {
	placeArray[idx] = new Place(iterator.next());
	//println("Place ["+idx+"]: "+ placeArray[idx].javaObject.getName());
    }

    return placeArray;
}


/**
 * Returns an array of all transitions inside this net.
 * 
 * @return An array of all this net's transitions.
 */
Net.prototype.getTransitions = function() {
    var iterator = this.javaObject.iterateTransitions();
    var transArray = new Array();

    for (var idx=0; iterator.hasNext(); idx++) {
	var javaTrans = iterator.next();
	if (javaTrans instanceof Packages.gpal.ImmediateTransition) {
	    transArray[idx] = new ImmediateTransition(javaTrans);
	} else if (javaTrans instanceof Packages.gpal.TimedTransition) {
	    transArray[idx] = new TimedTransition(javaTrans);
	} else if (javaTrans instanceof Packages.gpal.SubstitutionTransition) {
	    transArray[idx] = new SubstitutionTransition(javaTrans);
	} else if (javaTrans instanceof Packages.gpal.ModulePin) {
	    transArray[idx] = new ModulePin(javaTrans);
	} else {
	    session.throwException("Net.prototype.getTransition: Java Transition is of unknown type.");
	}
    }
    
    return transArray;
}


/**
 * Returns an array of all arcs inside this net.
 *
 * @return An array of all this net's arcs.
 */
Net.prototype.getArcs = function() {
    var iterator = this.javaObject.iterateArcs();
    var arcArray = new Array();

    for (var idx=0; iterator.hasNext(); idx++) {
	var javaArc = iterator.next();
	
	if (javaArc instanceof Packages.gpal.InputArc) {
	    arcArray[idx] = new InputArc(javaArc);
	} else if (javaArc instanceof Packages.gpal.OutputArc) {
	    arcArray[idx] = new OutputArc(javaArc);
	} else {
	    session.throwException("Net.prototype.getArcs: Java arc object is of unknown type.");
	}
    } 

    return arcArray;
}

