// -*- java -*-
//
// $Id: InputArc.js 10400 2007-08-07 23:38:00Z knoke $
//
// *************************************
//  InputArc
// *************************************

/**
 * Creates a new InputArc object.
 *
 * @param javaObject The encapsulated java object.
 */
function InputArc(javaObject) {
    if (javaObject == null) {
	session.throwException("InputArc constructor: provided javaObject == null!");
    }
    this.javaObject = javaObject;
}


/**
 * Adds a new InputArc to the given net and connects place and transition.
 *
 * @param net The net to add the input arc to.
 * @param place The place to start the input arc.
 * @param transition The Transition where this arc should end.
 * @param inscription The inscription of this arc.
 * @return The new InputArc.
 */
InputArc.add = function(net, place, transition, inscription) {
    if (net == null) {
	session.throwException("InputArc.add: provided net == null!");
    }
    
    if (! (place instanceof Place)) {
	session.throwException("InputArc.add: given place is not of type Place.");
    }

    if (! (transition instanceof ImmediateTransition) && 
	! (transition instanceof TimedTransition) && 
	! (transition instanceof SubstitutionTransition) && 
	! (transition instanceof ModulePin)) 
    {
	session.throwException("InputArc.add: given transition is not of type "
			       +"ImmediateTransition, SubstitutionTransition, TimedTransition"
			       +" or ModulePin.");
    }
    
    var javaArc = session.addInputArc(net.javaObject, place.javaObject, transition.javaObject, inscription); 
    return new InputArc(javaArc);
}

/** 
 * Tries to find an arc specified by its from- and to-node.
 *
 * @param net The net to search in.
 * @param place The place the arc starts from.
 * @param transition The transition the arc end in.
 * @param The arc, if found.
 */
InputArc.getFromTo = function(net, place, transition) {
    arcs = net.getArcs();
    var arc = null;
    
    if (net == null) {
	session.throwException("InputArc.add: provided net == null!");
    }
    
    if (! (place instanceof Place)) {
	session.throwException("InputArc.add: given place is not of type Place.");
    }

    if (! (transition instanceof ImmediateTransition) && 
	! (transition instanceof TimedTransition) && 
	! (transition instanceof SubstitutionTransition))
    {
	session.throwException("InputArc.getFromTo: given transition is not of type ImmediateTransition, SubstitutionTransition, TimedTransition");
    }
    
    for (k = 0; k < arcs.length; k++) {
        if(arcs[k] instanceof InputArc) {
            fplace = arcs[k].getPlace();
            ftransition = arcs[k].getTransition();
            if ((fplace.getID()).equals(place.getID()) && (ftransition.getID()).equals(transition.getID())) {
                arc = arcs[k];
                return arc;
            }
        }
    }
    
    if (arc == null) {
        session.throwException("InputArc.getFromTo: given arc could not be found.");
    }
}

/**
 * Sets the inscription of this arc.
 *
 * @param inscr The inscription to set.
 */
InputArc.prototype.setInscription = function(inscr) {
    this.javaObject.setInscriptionStr(inscr);
}

/**
 * Returns the inscription of this arc.
 */
InputArc.prototype.getInscription = function() {
    return this.javaObject.getInscriptionStr();
}

/**
 * Returns the place the arc starts from.
 */
InputArc.prototype.getPlace = function() {
    return new Place(this.javaObject.getPlace());
}

/**
 * Returns the transition the arc ends in.
 */
InputArc.prototype.getTransition = function() {
    if (this.javaObject.getTransition() instanceof Packages.gpal.TimedTransition)
        return new TimedTransition(this.javaObject.getTransition());
    if (this.javaObject.getTransition() instanceof Packages.gpal.ImmediateTransition)
        return new ImmediateTransition(this.javaObject.getTransition());
    if (this.javaObject.getTransition() instanceof Packages.gpal.SubstitutionTransition)
        return new SubstitutionTransition(this.javaObject.getTransition());
}