// -*- java -*-
//
// $Id: ImmediateTransition.js 10400 2007-08-07 23:38:00Z knoke $
//
// *************************************
//  Immediate Transition
// *************************************

/**
 * Creates a new ImmediateTransition object.
 * 
 * @param javaObject The Java object to ecapsulate.
 */
function ImmediateTransition(javaObject) {
    if (javaObject == null) {
	session.throwException("ImmediateTransition constructor: provided javaObject == null!");
    }
    this.javaObject = javaObject;
}


/**
 * Adds a new ImmediateTransition to the given net with the provided properties.
 *
 * @param net The Petri Net to add this transition to.
 * @param name The name of the new Immediatetransition.
 * @param priority The priority of the transition.
 * @param weight The weight of the Immediatetransition.
 * @return A new ImmediateTransition.
 */
ImmediateTransition.add = function(net, name, priority, weight) {
  if (net == null)
    session.throwException("No net for Immediate Transition!");
  if (name == null)
    session.throwException("No name for Immediate Transition!");
  if (weight == null)
    session.throwException("No weight for Immediate Transition!");
  
  return new ImmediateTransition(session.addImmediateTransition(net.javaObject, name, priority, weight));   
}


/**
 * Tries to find the ImmediateTransition with the given name in the provided net.
 *
 * @param net The net to search in.
 * @param name The name to use in search.
 * @return The ImmediateTransition, if found.
 */
ImmediateTransition.withName = function(net, name) {
  if (net == null)
    session.throwException("ImmediateTransition.withName: provided net == null!");
  if (name == null) {
    session.throwException("ImmediateTransition.withName: provided name == null!");
  }

  var trans = session.getImmediateTransitionWithName(net.javaObject, name);
  if(trans == null) {
    session.throwException("ImmediateTransition " + name + " could not be found!");
  }
  return new ImmediateTransition(trans);
}


/**
 * Returns the name of this ImmediateTransition.
 *
 * @return The name.
 */
ImmediateTransition.prototype.getName = function() {
    return this.javaObject.getName();
}


/**
 * Returns the ID of this ImmediateTransition.
 *
 * @return The ID.
 */
ImmediateTransition.prototype.getID = function() {
    return this.javaObject.getId();
}


/** 
 * Sets the local guard expression of this transition.
 * 
 * @param guardStr The guard expression.
 */
ImmediateTransition.prototype.setLocalGuard = function(guardStr) {
    this.javaObject.setLocalGuard(guardStr);
}


/**
 * Returns the local guard expression of this transition.
 *
 * @return The local guard expression.
 */
ImmediateTransition.prototype.getLocalGuard = function() {
    return this.javaObject.getLocalGuard().toString();
}


/**
 * Sets the global guard expression of this transition.
 * 
 * @param guardStr The guard expression.
 */
ImmediateTransition.prototype.setGlobalGuard = function(guardStr) {
    this.javaObject.setGlobalGuard(guardStr);
}


/**
 * Returns the global guard expression of this transition.
 *
 * @return The global guard expression.
 */
ImmediateTransition.prototype.getGlobalGuard = function() {
    return this.javaObject.getGlobalGuard().toString();
}


/**
 * Returns this transition's priority.
 *
 * @return The priority value.
 */
ImmediateTransition.prototype.getPriority = function() {
    return this.javaObject.getPriority().getExpressionStr();
}


/**
 * Sets the priority of this transition.
 *
 * @param priority The new priority value.
 */
ImmediateTransition.prototype.setPriority = function(priority) {
    this.javaObject.setPriority(priority);
}


/**
 * Returns this transition's weight.
 *
 * @return The weight value.
 */
ImmediateTransition.prototype.getWeight = function() {
    return this.javaObject.getWeight().getExpressionStr();
}


/**
 * Sets the weight of this ImmediateTransition.
 *
 * @param weight The new weight value.
 */ 
ImmediateTransition.prototype.setWeight = function(weight) {
    this.javaObject.setWeight(weight);
}


/**
 * Returns the value of this tranision's watch attribute.
 */
ImmediateTransition.prototype.getWatch = function() {
	return this.javaObject.getWatch();
}


/**
 * Sets the value of this transition's watch attribute.
 *
 * @param w The new value of the watch attribute.
 */
ImmediateTransition.prototype.setWatch = function(w) {
	this.javaObject.setWatch(w);
}


/**
 * Returns the value of this transition's SpecType attribute.
 */
ImmediateTransition.prototype.getSpecType = function() {
	return this.javaObject.getSpecType();
}


/**
 * Sets the value of this transition's SpecType attribute.
 *
 * @param spec The new value of the SpecType attribute
 */
ImmediateTransition.prototype.setSpecType = function(spec) {
	this.javaObject.setSpecType(spec);
}


/**
 * Sets the display expression.
 *
 * @param expression The new display expression to set.
 */
ImmediateTransition.prototype.setDisplayExpression = function(expression) {
    this.javaObject.setDisplayExpression(expression);
}


/**
 * Returns the value of the display expression.
 */
ImmediateTransition.prototype.getDisplayExpression = function() {
	return (this.javaObject.getDisplayExpression()).toString();
}


/**
 * Sets the server type.
 *
 * @param expression The new server type to set.
 */
ImmediateTransition.prototype.setServerType = function(expression) {
    this.javaObject.setServerType(expression);
}


/**
 * Returns the server type.
 */
ImmediateTransition.prototype.getServerType = function() {
	return this.javaObject.getServerType();
}


/**
 * Sets the value of the takeFirst attribute.
 *
 * @param takeFirst The new value of the takeFirst attribute.
 */
ImmediateTransition.prototype.setTakeFirst = function(takeFirst) {
	this.javaObject.setTakeFirst(takeFirst);
}


/**
 * Returns the value of the takeFirst attribute.
 */
ImmediateTransition.prototype.getTakeFirst = function() {
	return this.javaObject.getTakeFirst();
}


/** 
 * Sets this transition's manual code.
 */
ImmediateTransition.prototype.setManualCode = function(code) {
	this.javaObject.setManualCode(code);
}


/**
 * Returns this transition's manual code.
 */
ImmediateTransition.prototype.getManualCode = function() {
	return this.javaObject.getManualCode();
}


/**
 * Returns this transition's logfile name.
 */
ImmediateTransition.prototype.getLogfileName = function() {
	return this.javaObject.getLogfileName();
}


/**
 * Returns this tranistion's logfile description.
 */
ImmediateTransition.prototype.getLogfileDescription = function() {
	return this.javaObject.getLogfileDescription();
}


/**
 * Returns this transition's logfile expression.
 */
ImmediateTransition.prototype.getLogfileExpression = function() {
	return this.javaObject.getLogfileExpressionStr();
}


/**
 * Sets the parameters for this transition's logfile.
 */
ImmediateTransition.prototype.setLogfileParameters = function(filename, expression, description) {
	this.javaObject.setLogfileParameters(filename, expression, description);
}